import { Navbar } from "@/components/navbar"
import { Footer } from "@/components/footer"
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Progress } from "@/components/ui/progress"
import { Users, Calendar, PlusCircle, Search } from "lucide-react"
import Image from "next/image"
import Link from "next/link"

export default function ProjectHubs() {
  const projects = [
    {
      id: 1,
      title: "AI-Powered Study Assistant",
      description: "Developing an intelligent tutoring system that adapts to individual learning styles.",
      category: "Computer Science",
      members: 6,
      progress: 75,
      deadline: "Dec 15, 2024",
      status: "Active",
      image: "/images/ai-project.jpg",
      lead: "Sarah Johnson",
      leadAvatar: "/images/avatar-1.jpg",
    },
    {
      id: 2,
      title: "Sustainable Energy Solutions",
      description: "Research project on renewable energy implementation in urban environments.",
      category: "Environmental Science",
      members: 8,
      progress: 45,
      deadline: "Jan 20, 2025",
      status: "Active",
      image: "/images/energy-project.jpg",
      lead: "Miguel Rodriguez",
      leadAvatar: "/images/avatar-2.jpg",
    },
    {
      id: 3,
      title: "Digital Literature Archive",
      description: "Creating a comprehensive digital library of classical and contemporary literature.",
      category: "Literature",
      members: 4,
      progress: 90,
      deadline: "Nov 30, 2024",
      status: "Finishing",
      image: "/images/literature-project.jpg",
      lead: "Emma Lewis",
      leadAvatar: "/images/avatar-5.jpg",
    },
    {
      id: 4,
      title: "Chemistry Lab Simulator",
      description: "Virtual reality chemistry lab for safe experimentation and learning.",
      category: "Chemistry",
      members: 5,
      progress: 30,
      deadline: "Mar 15, 2025",
      status: "Planning",
      image: "/images/chemistry-project.jpg",
      lead: "Dr. Michael Brown",
      leadAvatar: "/images/avatar-4.jpg",
    },
  ]

  return (
    <div className="min-h-screen flex flex-col">
      <Navbar />

      <main className="flex-grow">
        <div className="bg-teal-500 text-white py-12 relative overflow-hidden">
          <div className="absolute inset-0 z-0">
            <Image
              src="https://hebbkx1anhila5yf.public.blob.vercel-storage.com/Project%20Hubs-3SrsY1xgDmkCHUD6c7uqiyat15f7a7.jpeg"
              alt="Project Hubs Background"
              fill
              className="object-cover opacity-20"
            />
          </div>
          <div className="container mx-auto px-4 relative z-10">
            <h1 className="text-3xl md:text-4xl font-bold mb-4">Project Hubs</h1>
            <p className="text-xl max-w-3xl">
              Collaborate on innovative projects, manage tasks, and bring your ideas to life with fellow students and
              educators.
            </p>
          </div>
        </div>

        <div className="container mx-auto px-4 py-8">
          <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4 mb-8">
            <div className="relative w-full md:w-96">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" size={18} />
              <Input placeholder="Search projects..." className="pl-10" />
            </div>

            <Button className="bg-teal-500 hover:bg-teal-600">
              <PlusCircle className="mr-2 h-4 w-4" />
              Create Project
            </Button>
          </div>

          <Tabs defaultValue="all" className="mb-8">
            <TabsList>
              <TabsTrigger value="all">All Projects</TabsTrigger>
              <TabsTrigger value="my-projects">My Projects</TabsTrigger>
              <TabsTrigger value="joined">Joined</TabsTrigger>
              <TabsTrigger value="completed">Completed</TabsTrigger>
            </TabsList>

            <TabsContent value="all" className="mt-6">
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {projects.map((project) => (
                  <Card key={project.id} className="overflow-hidden hover:shadow-lg transition-shadow">
                    <div className="relative h-48">
                      <Image
                        src={project.image || "/placeholder.svg"}
                        alt={project.title}
                        fill
                        className="object-cover"
                      />
                      <div className="absolute top-2 right-2 bg-white/90 text-teal-800 px-2 py-1 rounded text-xs font-medium">
                        {project.status}
                      </div>
                    </div>
                    <CardHeader>
                      <CardTitle className="text-lg">{project.title}</CardTitle>
                      <CardDescription>{project.category}</CardDescription>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <p className="text-sm text-gray-600 line-clamp-2">{project.description}</p>

                      <div className="flex items-center justify-between">
                        <div className="flex items-center">
                          <Avatar className="h-6 w-6 mr-2">
                            <AvatarImage src={project.leadAvatar || "/placeholder.svg"} alt={project.lead} />
                            <AvatarFallback>{project.lead.charAt(0)}</AvatarFallback>
                          </Avatar>
                          <span className="text-sm text-gray-600">{project.lead}</span>
                        </div>
                        <div className="flex items-center text-gray-500">
                          <Users className="h-4 w-4 mr-1" />
                          <span className="text-sm">{project.members}</span>
                        </div>
                      </div>

                      <div>
                        <div className="flex items-center justify-between mb-1">
                          <span className="text-sm text-gray-600">Progress</span>
                          <span className="text-sm font-medium">{project.progress}%</span>
                        </div>
                        <Progress value={project.progress} className="h-2" />
                      </div>

                      <div className="flex items-center text-gray-500">
                        <Calendar className="h-4 w-4 mr-1" />
                        <span className="text-sm">Due: {project.deadline}</span>
                      </div>
                    </CardContent>
                    <CardFooter>
                      <Button asChild className="w-full bg-teal-500 hover:bg-teal-600">
                        <Link href={`/projects/${project.id}`}>View Project</Link>
                      </Button>
                    </CardFooter>
                  </Card>
                ))}
              </div>
            </TabsContent>

            <TabsContent value="my-projects">
              <div className="py-12 text-center text-gray-500">Sign in to view your projects</div>
            </TabsContent>

            <TabsContent value="joined">
              <div className="py-12 text-center text-gray-500">Sign in to view projects you've joined</div>
            </TabsContent>

            <TabsContent value="completed">
              <div className="py-12 text-center text-gray-500">Sign in to view completed projects</div>
            </TabsContent>
          </Tabs>

          <div className="bg-gray-50 rounded-xl p-6 mt-12">
            <div className="flex flex-col md:flex-row items-start md:items-center gap-6">
              <div className="relative w-full md:w-1/3 h-48 rounded-lg overflow-hidden">
                <Image
                  src="https://hebbkx1anhila5yf.public.blob.vercel-storage.com/Project%20Hubs-3SrsY1xgDmkCHUD6c7uqiyat15f7a7.jpeg"
                  alt="Project Collaboration"
                  fill
                  className="object-cover"
                />
              </div>
              <div className="md:w-2/3">
                <h2 className="text-2xl font-bold mb-2">Start Your Next Big Project</h2>
                <p className="text-gray-600 mb-4">
                  Have an innovative idea? Create a project hub, invite collaborators, and turn your vision into
                  reality. Use our project management tools to track progress and achieve your goals.
                </p>
                <div className="flex flex-wrap gap-4">
                  <Button className="bg-teal-500 hover:bg-teal-600">Create Project</Button>
                  <Button variant="outline">Browse Templates</Button>
                </div>
              </div>
            </div>
          </div>
        </div>
      </main>

      <Footer />
    </div>
  )
}
